
import {
    userDb,
    createDefaultSubscription,
    checkEmailLimit,
    subscriptionPlansDb,
    userSubscriptionsDb,
    packagesDb
} from '../src/lib/database.js';
import { neon } from '@neondatabase/serverless';

if (!process.env.DATABASE_URL) {
    console.error('DATABASE_URL is not set.');
    process.exit(1);
}

const sql = neon(process.env.DATABASE_URL);

async function runTest() {
    console.log('🧪 Starting Webhook Logic Reproduction Test...');

    const uniqueId = Math.floor(Math.random() * 100000);
    const testUser = {
        name: `Webhook Test ${uniqueId}`,
        email: `webhook_test_${uniqueId}@example.com`,
        company: 'Test Corp',
        job_title: 'Tester'
    };

    try {
        // 1. Create User & Subscription
        console.log(`\n1️⃣ Creating user and subscription...`);
        const { lastInsertRowid: userId } = await userDb.create(testUser);

        // IMPORTANT: Move to PRO plan first (simulating user who upgraded)
        const proPlan = await subscriptionPlansDb.getByName('Pro');
        if (!proPlan) throw new Error('Pro plan not found');
        await userSubscriptionsDb.create(userId, proPlan.id);
        console.log(`   User created (ID: ${userId}) and set to Pro plan.`);

        // 2. Check initial limit (should be blocked as no package purchased)
        let limitCheck = await checkEmailLimit(userId, 'generation');
        console.log(`   Initial Check: Allowed=${limitCheck.allowed}, Reason=${limitCheck.reason}`);

        if (limitCheck.allowed) {
            throw new Error('Pro user without package should initially be blocked.');
        }

        // 3. Simulate CURRENT Webhook Logic (Update `users` table ONLY)
        console.log(`\n2️⃣ Simulating CURRENT Webhook Logic (updating 'users' table)...`);
        await sql`
        UPDATE users 
        SET 
          generations_remaining = 100,
          current_package = 'Test Package'
        WHERE id = ${userId}
    `;
        console.log('   Updated users table directly.');

        // 4. Check limit again (EXPECT FAILURE)
        limitCheck = await checkEmailLimit(userId, 'generation');
        console.log(`   Check after Webhook: Allowed=${limitCheck.allowed}, Reason=${limitCheck.reason}`);

        if (limitCheck.allowed) {
            console.log("   ❌ SURPRISE: Limit IS allowed. 'users' table IS used?");
        } else {
            console.log("   ✅ CONFIRMED: Updating 'users' table DOES NOT START THE PACKAGE.");
            console.log('      The webhook logic is indeed insufficient.');
        }

        // 5. Simulate CORRECT Logic (using purchasePackage)
        console.log(`\n3️⃣ Simulating CORRECT Logic (using userSubscriptionsDb.purchasePackage)...`);
        // Need a valid package ID. Let's find one.
        const packages = await packagesDb.getAll();
        if (packages.length === 0) throw new Error('No packages found in DB');
        const pkg = packages[0];

        await userSubscriptionsDb.purchasePackage(userId, pkg.id);
        console.log(`   Called purchasePackage for package: ${pkg.name}`);

        // 6. Check limit again (EXPECT SUCCESS)
        limitCheck = await checkEmailLimit(userId, 'generation');
        console.log(`   Check after Correct Logic: Allowed=${limitCheck.allowed}, Remaining=${limitCheck.remaining}`);

        if (limitCheck.allowed) {
            console.log('   ✅ SUCCESS: purchasePackage correctly activated the subscription.');
        } else {
            throw new Error('purchasePackage failed to activate subscription.');
        }

        console.log('\n✅ REPRODUCTION COMPLETED');
        process.exit(0);

    } catch (error) {
        console.error('\n❌ TEST FAILED:', error);
        process.exit(1);
    }
}

runTest();
